/* These are the routines for handling signals delivered to emulated
 * programs.
 *
 * Handling signals for an emulated program is tricky.  The signal is
 * generated by the host sytem in a form that programs on the host
 * system can understand.  The signal handler must translate this
 * signal into a form that can be understood by the emulated program.
 * Any actions taken *by* the emulated program in response to this
 * signal must also be propagated back to the mainline program.
 *
 * This whole thing is further complicated by those times when we
 * are *not* emulating Intel instructions (e.g. in a system call or
 * in a native shared library or whatnot).
 *
 * For now, in a pure-emulated environment, we can handle things as
 * follows:
 *
 *	Signals generated *outside* the program (e.g. SIGINT, SIGALRM)
 *	are received and noted, but processing them is deferred
 *	until the end of the instruction being emulated.  This way,
 * 	we don't have to worry about restarting or aborting
 *	partially-comlete instructions.
 *
 *	Signals generated *by* the program (e.g. SIGSEGV, SIGILL,
 *	SIGFPE) are handled synchronously by the emulator via a
 *	special entrypoint to the signal processor.
 */

#include <stdio.h>
#include <signal.h>
#include <sigcontext.h>
#include <errno.h>
#include <syscall_thunk.h>
#include <setjmp.h>
#include <sys/mman.h>
#include "linux_bin.h"
#include "x86_cpu.h"

extern int die_on_segv;
extern FILE * x86_logfile;
extern X86_CPU *	current_cpu;
extern unsigned long stktop;

#define CPU_HEAP_START 0x79000000

/* Maps to go from x86 to alpha signal numbers and back again */

static int x86_to_alpha_signum[] = { 0, 1, 2, 3, 4, 5, 6, 10, 8, 9, 30, 11, 31,
				13, 14, 15, 7, 20, 19, 17, 18, 21,
				22, 16, 24, 25, 26, 27, 28, 23, 29, 12 };

static int alpha_to_x86_signum[] = { 0, 1, 2, 3, 4, 5, 6, 16, 8, 9, 7, 11,
				31, 13, 14, 15, 23, 19, 20, 18, 17, 21, 22,
				29, 24, 25, 26, 27, 28, 30, 10, 12 };

static bitmask_transtbl sigmask_tbl[] = {
	{ 0x00000001, 0x00000001, 0x00000001, 0x00000001 },
	{ 0x00000002, 0x00000002, 0x00000002, 0x00000002 },
	{ 0x00000004, 0x00000004, 0x00000004, 0x00000004 },
	{ 0x00000008, 0x00000008, 0x00000008, 0x00000008 },
	{ 0x00000010, 0x00000010, 0x00000010, 0x00000010 },
	{ 0x00000020, 0x00000020, 0x00000020, 0x00000020 },
	{ 0x00000040, 0x00000040, 0x00000200, 0x00000200 },
	{ 0x00000080, 0x00000080, 0x00000080, 0x00000080 },
	{ 0x00000100, 0x00000100, 0x00000100, 0x00000100 },
	{ 0x00000200, 0x00000200, 0x20000000, 0x20000000 },
	{ 0x00000400, 0x00000400, 0x00000400, 0x00000400 },
	{ 0x00000800, 0x00000800, 0x40000000, 0x40000000 },
	{ 0x00001000, 0x00001000, 0x00001000, 0x00001000 },
	{ 0x00002000, 0x00002000, 0x00002000, 0x00002000 },
	{ 0x00004000, 0x00004000, 0x00004000, 0x00004000 },
	{ 0x00008000, 0x00008000, 0x00000040, 0x00000040 },
	{ 0x00010000, 0x00010000, 0x00080000, 0x00080000 },
	{ 0x00020000, 0x00020000, 0x00040000, 0x00040000 },
	{ 0x00040000, 0x00040000, 0x00010000, 0x00010000 },
	{ 0x00080000, 0x00080000, 0x00020000, 0x00020000 },
	{ 0x00100000, 0x00100000, 0x00100000, 0x00100000 },
	{ 0x00200000, 0x00200000, 0x00200000, 0x00200000 },
	{ 0x00400000, 0x00400000, 0x00008000, 0x00008000 },
	{ 0x00800000, 0x00800000, 0x00800000, 0x00800000 },
	{ 0x01000000, 0x01000000, 0x01000000, 0x01000000 },
	{ 0x02000000, 0x02000000, 0x02000000, 0x02000000 },
	{ 0x04000000, 0x04000000, 0x04000000, 0x04000000 },
	{ 0x08000000, 0x08000000, 0x08000000, 0x08000000 },
	{ 0x10000000, 0x10000000, 0x00400000, 0x00400000 },
	{ 0x20000000, 0x20000000, 0x10000000, 0x10000000 },
	{ 0x40000000, 0x40000000, 0x00000800, 0x00000800 },
	{ 0, 0, 0, 0 }
};

static char *	x86_signal_names[] = { 
				"SIGNAL_0", "SIGHUP", "SIGINT", "SIGQUIT",
				"SIGILL", "SIGTRAP", "SIGABRT", "SIGBUS",
				"SIGFPE", "SIGKILL", "SIGUSR1", "SIGSEGV",
				"SIGUSR2", "SIGPIPE", "SIGALRM", "SIGTERM",
				"SIGSTKFLT", "SIGCHLD", "SIGCONT", "SIGSTOP",
				"SIGTSTP", "SIGTTIN", "SIGTTOU", "SIGURG",
				"SIGXCPU", "SIGXFSZ", "SIGVTALRM", "SIGPROF",
				"SIGWINCH", "SIGIO", "SIGPWR", "SIGUNUSED" };


/* Store the sigactions in user space because that's where we're
 * going to dispatch them from...  Also store some additional
 * handling information with each sigaction...
 */
struct em86_sigaction {
    struct x86_sigaction	sa;
    unsigned long		oldmask;
    struct sigcontext		sc;
    int				traced;
};

static struct em86_sigaction	sigact_table[NSIG];

/* This is a mask of pending signals.  We use this to perform
 * slightly-deferred signal delivery (i.e. between emulated machine 
 * instructions).
 */
unsigned int		x86_sigmask = 0;

/* For debugging... */
static int			x86_signal_count = 0;

#define BLOCKABLE	~((1<<(SIGKILL-1))|(1<<(SIGSTOP-1)))

/* Handle signals as they come in.  Basically, we just note that the
 * signal happened.  Another function will be called between emulated
 * instructions to actually take care of signals... 
 */
extern "C" void	
host_signal_handler(int signum, int code, struct sigcontext *scp)
{
    int		x86_signum;

    x86_signum = alpha_to_x86_signum[signum];

    /* If this is a seg fault in the stack area, then grow the stack and
     * return.
     */

    if((signum == SIGSEGV) &&
       (scp->sc_traparg_a0 > ABSOLUTE_STACK_BOTTOM) &&
       (scp->sc_traparg_a0 < stktop)) {

	grow_stack(scp->sc_traparg_a0);
        return;
    }

    if((signum == SIGSEGV) && die_on_segv) {
	fprintf(x86_logfile, "pid %d: SIGSEGV received\n",
			getpid());
	exit(-1);
    }
    if(sigact_table[x86_signum].sa.sa_handler == X86_SIG_DFL) {
	/* No signal handler declared; we deal with it ourselves */
	if((signum != SIGCHLD) && (signum != SIGURG) && (signum != SIGWINCH)) {
	    fprintf(x86_logfile, "Signal %d received\n", signum);
	    exit(-signum);
	}
    } else {
	unsigned long	alpha_mask;

	/* queue the signal for later processing... */
        x86_sigmask |= (1 << x86_signum);
	sigact_table[x86_signum].sc = *scp;

	/* Make sure we don't get this signal again until we've
	 * handled it...
	 */
	alpha_mask = x86_to_alpha_bitmask(sigact_table[x86_signum].sa.sa_mask, sigmask_tbl);
	if(!(sigact_table[x86_signum].sa.sa_flags & X86_SA_NOMASK)) {
		alpha_mask |= (1 << (signum-1));
	}

	/* Synchronous faults must be handled right away... */
	if((signum == SIGSEGV) || (signum == SIGILL)) {
	    x86_process_pending_signals();
	}
	else {

	    /* We'll process the signal later.  Modify the sigcontext so that 
	     * on return from this signal handler the new mask will be 
	     * installed...
	     */
	    sigact_table[x86_signum].oldmask = scp->sc_mask;
	    scp->sc_mask = alpha_mask & BLOCKABLE;
	}
    }
}

/* Initialize the signal handling.  We set all of the *system* signal
 * handlers to point to our signal handler.  We then keep track of the
 * emulated program's signal handlers ourselves.
 */
extern "C" void	x86_signal_handler_init()
{
    int		i;
    struct sigaction	sa;

    /* All system handlers point to our handler */
    sa.sa_handler = (void *)host_signal_handler;
    sa.sa_mask = 0;
    sa.sa_flags = 0;
    for(i = 0; i < NSIG; i++) {
	sigaction(i, &sa, NULL);
    }

    /* All emulated handlers are default */
    for(i = 0; i < NSIG; i++) {
	sigact_table[i].sa.sa_handler = X86_SIG_DFL;
	sigact_table[i].sa.sa_mask = 0;
	sigact_table[i].sa.sa_flags = 0;
	sigact_table[i].oldmask = 0;
	sigact_table[i].traced = 0;
    }

    /* Are we going to trace signals ? */
    if(getenv("X86_TRACE_SIGNALS")) {
	char *	signame;
	char	envstring[512];
	int	sigstate;

	strcpy(envstring, getenv("X86_TRACE_SIGNALS"));
	signame = strtok(envstring, ",");
	while(signame) {
	    if(*signame == '-') {
		sigstate = 0;
		signame++;
	    }
	    else {
		sigstate = 1;
	    }
	    if(strcmp(signame, "all") == 0) {
		for(i = 0; i < NSIG; i++) {
		    sigact_table[i].traced = sigstate;
		}
	    }
	    else {
		for(i = 0; i < NSIG; i++) {
		    if(strcmp(x86_signal_names[i], signame) == 0) {
			sigact_table[i].traced = sigstate;
			break;
		    }
		}
	    }
	    signame = strtok(NULL, ",");
	}
    }
}

extern "C" void	x86_process_pending_signals()
{
    int		i;
    int		signum;
    int		handler_signal_count;
    int			j;
    unsigned int *frame;

    if(x86_sigmask) {
	for(i = 1; i < NSIG; i++) {
	    if(x86_sigmask & (1 << i)) {
		x86_sigmask &= ~(1 << i);	/* Okay, we got it! */
		signum = i;
		switch(sigact_table[signum].sa.sa_handler) {
		    case X86_SIG_DFL:
			fprintf(x86_logfile, 
			  "Signal %s received; program terminated\n", 
			  x86_signal_names[signum]);
			em86_exit();
			exit(-1);
		    case X86_SIG_IGN:
			/* Quietly make this go away and go about our
			 * business...
			 */
			x86_sigmask &= ~(1 << i);
			break;
		    default: {
			/* Instantiate a new CPU to run the signal handler, as well
			 * as a jmp_buf to get us back here when we're done...
			 */
			X86_CPU  	sig_cpu;
			volatile X86_CPU  *	prev_cpu;
			volatile jmp_buf	*	jbp;
			volatile char *		stkp;
			volatile unsigned int	eax, ebx, ecx, edx;
			volatile unsigned int	esi, edi, ebp, esp, eip;

			/* Materialize the emulated registers */
			if(current_cpu->registers_homed()) {
				eax = current_cpu->get_EAX();
				ebx = current_cpu->get_EBX();
				ecx = current_cpu->get_ECX();
				edx = current_cpu->get_EDX();
				esi = current_cpu->get_ESI();
				edi = current_cpu->get_EDI();
				ebp = current_cpu->get_EBP();
				esp = current_cpu->get_ESP();
				eip = current_cpu->get_EIP();
			}
			else {
#ifdef USE_FX_CPU
				/* This is a bit ugly, as it relies on knowledge of
				 * FX emulator internals to know where the
				 * registers are...
				 */
				eax=sigact_table[signum].sc.sc_regs[0]; //eax
				ebx=sigact_table[signum].sc.sc_regs[9]; //ebx
				ecx=sigact_table[signum].sc.sc_regs[7]; //ecx
				edx=sigact_table[signum].sc.sc_regs[8]; //edx
				esi=sigact_table[signum].sc.sc_regs[10]; //esi
				edi=sigact_table[signum].sc.sc_regs[11]; //edi
				ebp=sigact_table[signum].sc.sc_regs[12]; //ebp
				esp=sigact_table[signum].sc.sc_regs[13]; //esp
				eip=sigact_table[signum].sc.sc_regs[14];//eip
#else
				fprintf(x86_logfile, "PANIC: Registers not homed in BOCHS CPU!\n");
				exit(-1);
#endif
			}


			/* Push the jmp_buf onto the stack... */
			stkp = (char *)esp;
			stkp = (char *)(((unsigned long)stkp - sizeof(jmp_buf)) & ~7);
			jbp = (jmp_buf *)stkp;

			if(setjmp(*jbp) == 0) {
			    /* Dispatch path.  Push the trampoline code onto the stack
			     * and fire up the new cpu.
			     */
			    frame = (unsigned int *)stkp;

			    handler_signal_count = ++x86_signal_count;
			    if(sigact_table[signum].traced) {
				fprintf(x86_logfile, 
				    "Processing %s, count %d, dispatching to signal handler @0x%x\n", 
				    x86_signal_names[signum], handler_signal_count, sigact_table[signum].sa.sa_handler);
				fprintf(x86_logfile, "EAX = 0x%x, EBX = 0x%x, ECX = 0x%x, EDX = 0x%x\n",
						eax, ebx, ecx, edx);
				fprintf(x86_logfile, "ESI = 0x%x, EDI = 0x%x, EBP = 0x%x, ESP = 0x%x\n",
						esi, edi, ebp, esp);
				fprintf(x86_logfile, "EIP = 0x%x\n", eip);
			        fflush(x86_logfile);
			    }

			    /* Build a Linux/Intel-style signal frame... */
			    frame -= 64;
			    frame[0] = (unsigned int)(&frame[24]) + 3;	/* Return addr of signal trampoline code */
			    if(sigact_table[signum].traced) {
				fprintf(x86_logfile, "Signal trampoline code @0x%x\n", frame[0]);
			    }
			    frame[1] = i;			/* Signal number */
			    frame[2] = 0;			/* gs (segment registers are unused) */
			    frame[3] = 0;			/* fs */
			    frame[4] = 0;			/* es */
			    frame[5] = 0;			/* ds */

			    frame[6] = edi;
			    frame[7] = esi;
			    frame[8] = ebp;
			    frame[9] = esp;
			    frame[10]= ebx;
			    frame[11]= edx;
			    frame[12]= ecx;
			    frame[13]= eax;
			    frame[16]= eip;

			    frame[14]= 0;			/* tss trap number */
			    frame[15]= 0;			/* tss error code */
			    frame[17]= 0;			/* cs */
			    frame[18]= 0;			/* EFLAGS - JRP may want to materialize this */
			    frame[19]= esp;
			    frame[20]= 0;			/* ss */
			    frame[21]= 0;			/* JRP - Should be pointer to i387 state */
			    frame[22]= 0;			/* Should be oldmask */
			    frame[23]= 0;			/* Intel puts CR2 here */
							/* Trampoline code follows: */
			    frame[24]= 0xea000000;	/* far "jmp" instruction. */
			    frame[25]= (unsigned int)jbp;		/* jump_buf pointer */
			    frame[26] = 0x00000002;	/* "segment" 2 */

			    for(j = 27; j < 64; j++) {
				frame[j] = 0xdead0000 + j + 1;
			    }

			    if(sigact_table[signum].traced) {
				fprintf(x86_logfile, "Signal handler SP @0x%x\n", frame);
			    }

			    sig_cpu.init_cpu(0, // eax
						0, //ebx
						0, //ecx
						0, //edx
						0, //esi
						0, //edi
						0, //ebp
						(int)frame, //esp
						(int)sigact_table[signum].sa.sa_handler// eip
					);
			    prev_cpu = current_cpu;
			    current_cpu = &sig_cpu;
			    /* Trace only the signal handler... */
#if 0
			    if(sigact_table[signum].traced) {
				current_cpu->bx_dbg.debugger = 1;
			    }
#endif 
			    sig_cpu.cpu_loop();
			    fprintf(x86_logfile, "PANIC: Returned from cpu_loop!\n");
			    exit(-1);
			}
			else {
			    /* Return path.  Go back to the previous CPU and continue
			     * as though nothing happened.  Later on we'll handle
			     * cases where the signal handler modified the sigcontext...
			     */
			    if(sigact_table[signum].traced) {
				int	stack_ok = 1;
				fprintf(x86_logfile,
					"Returned from %s handler, count=%d\n", 
					x86_signal_names[signum], handler_signal_count);
			        fflush(x86_logfile);
			    }
			    if(frame[13] != eax) {
				fprintf(x86_logfile, "signal handler modified EAX (0x%x -> 0x%x)\n", frame[13], eax);
			    }
			    if(frame[10] != ebx) {
				fprintf(x86_logfile, "signal handler modified EBX (0x%x -> 0x%x)\n", frame[10], ebx);
			    }
			    if(frame[12] != ecx) {
				fprintf(x86_logfile, "signal handler modified ECX (0x%x -> 0x%x)\n", frame[12], ecx);
			    }
			    if(frame[11] != edx) {
				fprintf(x86_logfile, "signal handler modified EDX (0x%x -> 0x%x)\n", frame[11], edx);
			    }
			    if(frame[7] != esi) {
				fprintf(x86_logfile, "signal handler modified ESI (0x%x -> 0x%x)\n", frame[7], esi);
			    }
			    if(frame[6] != edi) {
				fprintf(x86_logfile, "signal handler modified EDI (0x%x -> 0x%x)\n", frame[6], edi);
			    }
			    if(frame[8] != ebp) {
				fprintf(x86_logfile, "signal handler modified EBP (0x%x -> 0x%x)\n", frame[8], ebp);
			    }
			    if(frame[9] != esp) {
				fprintf(x86_logfile, "signal handler modified ESP (0x%x -> 0x%x)\n", frame[9], esp);
			    }
			    if(frame[16] != eip) {
				fprintf(x86_logfile, "signal handler modified EIP (0x%x -> 0x%x)\n", frame[16], eip);
			    }
			    current_cpu = prev_cpu;
			}
		    }
		}

		/* Now that we've handled the signal, restore the signal
		 * mask to what it was before we were so rudely 
		 * interrupted...
		 */
		sigprocmask(SIG_SETMASK, &(sigact_table[signum].oldmask), NULL);

	    }
	}
    }
}

/* And now, the signal-related system calls... */
extern "C" int do_sigaction(int a0, int a1, int a2, int a3, int a4, int a5)
{
    int		signum = a0;
    struct x86_sigaction *act = ARG_TO_POINTER(a1);
    struct x86_sigaction *oldact = ARG_TO_POINTER(a2);

    if(signum < 0 || signum >= NSIG) {
	return(-EINVAL);
    }

    if(oldact) {
	oldact->sa_handler = sigact_table[signum].sa.sa_handler;
	oldact->sa_mask = sigact_table[signum].sa.sa_mask;
	oldact->sa_flags = sigact_table[signum].sa.sa_flags;
    }

    if(act) {
	sigact_table[signum].sa.sa_handler = act->sa_handler;
	sigact_table[signum].sa.sa_mask = act->sa_mask;
	sigact_table[signum].sa.sa_flags = act->sa_flags;
    }

    return(0);
}

extern "C" int do_signal(int a0, int a1, int a2, int a3, int a4, int a5, int a6)
{
    int	signum = a0;
    unsigned int handler = a1;
    int old_handler;

    old_handler = sigact_table[signum].sa.sa_handler;

    sigact_table[signum].sa.sa_handler = handler;
    sigact_table[signum].sa.sa_mask = 0;
    sigact_table[signum].sa.sa_flags = X86_SA_ONESHOT;

    return(old_handler);
}

extern "C" int do_sigprocmask(int a0, int a1, int a2, int a3, int a4, int a5)
{
    int	x86_how = a0;
    unsigned int * x86_setp = ARG_TO_POINTER(a1);
    unsigned int * x86_osetp = ARG_TO_POINTER(a2);
    int how;
    unsigned long set;
    unsigned long oset;
    int retval;

    /* JRP - we may have to squirrel away our own copy of the signal
     * mask if we ever special-case any signals...
     */

    switch(x86_how) {
    	case X86_SIG_BLOCK:
	    how = SIG_BLOCK;
	    break;
	case X86_SIG_UNBLOCK:
	    how = SIG_UNBLOCK;
	    break;
	case X86_SIG_SETMASK:
	    how = SIG_SETMASK;
	    break;
	default:
	    return(-EINVAL);
    }

    if(x86_setp) {
        set = (unsigned long)x86_to_alpha_bitmask(*x86_setp, sigmask_tbl);
        retval = sigprocmask(how, (sigset_t *)&set, (sigset_t *)&oset);
    }
    else {
        retval = sigprocmask(how, NULL, (sigset_t *)&oset);
    }

    if(retval == 0) {
	if(x86_osetp) {
	    *x86_osetp = (unsigned int)alpha_to_x86_bitmask(oset, sigmask_tbl);
	}
        return(0);
    }
    else {
	return(-errno);
    }
}

extern "C" int do_ssetmask(int a0, int a1, int a2, int a3, int a4, int a5)
{
    unsigned long	mask;
    unsigned long	omask;
    int			retval;

    mask = (unsigned long)x86_to_alpha_bitmask(a0, sigmask_tbl);

    retval = sigprocmask(SIG_SETMASK, (sigset_t)&mask, (sigset_t)&omask);
    omask = (unsigned long)alpha_to_x86_bitmask(omask, sigmask_tbl);
    return((unsigned int)omask);
}

extern "C" int do_sgetmask(int a0, int a1, int a2, int a3, int a4, int a5)
{
    unsigned long	mask;
    unsigned long	omask;
    int			retval;

    mask = 0;

    retval = sigprocmask(SIG_BLOCK, (sigset_t)&mask, (sigset_t)&omask);
    omask = (unsigned long)alpha_to_x86_bitmask(omask, sigmask_tbl);
    return((unsigned int)omask);
}

extern "C" int do_sigsuspend(int a0, int a1, int a2, int a3, int a4, int a5)
{
    unsigned long	mask;
    int retval;

    mask = x86_to_alpha_bitmask(a0, sigmask_tbl);

    retval = sigsuspend(&mask);
    if(retval == 0) {
    	return(0);
    }
    else {
	return(-errno);
    }
}

extern "C" int do_sigpending(int a0, int a1, int a2, int a3, int a4, int a5)
{
    int retval;
    unsigned long mask;
    unsigned int *x86_maskp = ARG_TO_POINTER(a0);

    retval = sigpending(&mask);
    if(retval == 0) {
        *x86_maskp = alpha_to_x86_bitmask(mask, sigmask_tbl);
	return(0);
    }
    else {
        return(-errno);
    }
}
