#include <sys/types.h>
#include <sys/mman.h>
#include <signal.h>
#include "elf.h"
#include "linux_bin.h"
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>

#include "x86_cpu.h"


#define ALPHA_PAGE_SIZE	0x2000
#define ALPHA_PAGE_MASK 0xffffe000
/* This was 0xc0000000, but we ran into sign-extension problems in the
 * FX emulator....
 */
#define INITIAL_SP	0x80000000
#define INITIAL_STACK_SIZE	0x40000

FILE * x86_logfile;
X86_CPU *	current_cpu;

int	debug_mmap = 0;
int	die_on_segv = 0;

extern "C" {
    int elf_exec(char *, char **, char **, struct pt_regs *, struct image_info *);
    void x86_set_brk(char *);
    void x86_syscall_debug(int);
    void x86_signal_handler_init();
    void x86_process_pending_signals();
}

void sig_handler(int);

unsigned long	stktop;

main(int argc, char ** argv, char ** envp)
{
    X86_CPU	x86_cpu;
    struct pt_regs regs;
    struct image_info info;
    char *	syscall_trace;

    if(argc < 2) {
	printf("Usage: %s executable-file [args ...]\n", argv[0]);
	exit(0);
    }

    if(strcmp(argv[1], "--help") == 0) {
	printf("Usage: %s executable-file [args ...]\n", argv[0]);
	printf("The following environment variables can be used to control execution:\n");
	printf("    X86_TRACE_SYSCALLS - Traces system calls\n");
	printf("        e.g. X86_TRACE_SYSCALLS=open,read,write\n");
	printf("             X86_TRACE_SYSCALLS=all,-select\n");
	printf("    X86_TRACE_SIGNALS - Traces signal delivery\n");
	printf("        e.g. X86_TRACE_SIGNALS=SIGSEGV,SIGINT\n");
	printf("             X86_TRACE_SIGNALS=all,-SIGALRM\n");
	printf("    X86_SEGV_FATAL - Die if any SIGSEGV received\n");
	printf("    X86_DEBUG_MMAP - Traces memory-space management\n");
	printf("    X86_LOGFILE - File to receive all output from em86\n");
#ifdef USE_FX_CPU
	printf("    FX_INDEX_FILE - Location of the 'linux486.idx' file\n");
#endif
#ifdef USE_BOCHS_CPU
	printf("    BOCHS_DEBUG_CPU - Trace all x86 instructions\n");
#endif
	exit(0);
    }

    /* Zero out regs */
    memset(&regs, 0, sizeof(struct pt_regs));

    /* Zero out image_info */
    memset(&info, 0, sizeof(struct image_info));

    if(getenv("X86_DEBUG_MMAP")) {
	debug_mmap = 1;
    }

    if(getenv("X86_SEGV_FATAL")) {
	die_on_segv = 1;
    }

    if(getenv("X86_LOGFILE")) {
	x86_logfile = fopen(getenv("X86_LOGFILE"), "a");
	if(x86_logfile == NULL) {
	    fprintf(stderr, "Cannot open %s as logfile\n", 
			getenv("X86_LOGFILE"));
	    x86_logfile = stderr;
	}
	x86_cpu.set_logfile(x86_logfile);
    }
    else {
	x86_logfile = stderr;
	x86_cpu.set_logfile(x86_logfile);
    }

    /* Enable syscall debugging if desired... */
    syscall_trace = getenv("X86_TRACE_SYSCALLS");
    if(syscall_trace) {
	char	*syscall_name;

	syscall_name = strtok(syscall_trace, ",");
	while(syscall_name) {
	    if(strcmp(syscall_name, "all") == 0) {
		x86_syscall_trace_all();
	    }
	    else {
		if(*syscall_name == '-') {
		    x86_syscall_untrace(syscall_name+1);
		}
		else {
		    x86_syscall_trace(syscall_name);
		}
	    }
	    syscall_name = strtok(NULL, ",");
	}
    }


    if(elf_exec(argv[1], argv+1, envp, &regs, &info) != 0) {
	printf("Error loading %s\n", argv[1]);
	exit(-1);
    }

    x86_set_brk(info.brk);

    x86_signal_handler_init();


    /* Okay; program is loaded... now try to run it! */
    x86_cpu.init_cpu( regs.eax, regs.ebx, regs.ecx, regs.edx,
    		      regs.esi, regs.edi, regs.ebp, regs.esp,
		      regs.eip);
    current_cpu = &x86_cpu;
    x86_cpu.cpu_loop();
}

unsigned long instruction_count = 0;

extern "C" {
    void x86_process_pending_signals();
}

/* This routine is called by the Bochs emulator after every
 * instruction.  In addition to keeping statistics, we'll use
 * this as a useful hook to do deferred signal processing....
 */
#ifdef USE_BOCHS_CPU
void bx_instruction_tick(unsigned int eip)
{
    instruction_count++;
    x86_process_pending_signals();
}
#endif

extern "C" void em86_exit(void)
{
#ifdef USE_BOCHS_CPU
    printf("%d instructions emulated\n", instruction_count);
#endif
}

